﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace wol
{
	public static class OwlNameInSpace
	{

		public const string NamespaceDelimiter = "#";
		public const string NamespaceLeftBracket = "{";
		public const string NamespaceRightBracket = "}";

		/// <summary>
		/// Given a namespace name, return the same name, assured to be inside curly brackets.
		/// </summary>
		/// <param name="source">a namespace name that might or might not be inside curly brackets</param>
		/// <returns>the same namespace name, assured to be inside curly brackets</returns>
		public static string NormalizeNamespace(string source)
		{
			string result = source;
			if (false == result.StartsWith(NamespaceLeftBracket))
			{
				result = NamespaceLeftBracket + result;
			}
			if (false == result.EndsWith(NamespaceRightBracket))
			{
				result = result + NamespaceRightBracket;
			}
			return result;
		}

		/// <summary>
		/// Given a name and knowledge about namespaces, return a full namespace-qualified name.
		/// </summary>
		/// <param name="source">the name to be normalized</param>
		/// <param name="inModel">the knowledge about namespaces</param>
		/// <returns>the name, assured to be prefixed with a normal namespace name</returns>
		public static string NormalizeName(string source, OwlModel inModel)
		{
			string sourceValue = null;
			string sourceNamespace = null;
			OwlNameInSpace.DecodeString(source, inModel, out sourceNamespace, out sourceValue);
			string result = sourceNamespace + sourceValue;
			return result;
		}

		/// <summary>
		/// Given a name, which may be prefixed with a namespace name, return just the name.
		/// </summary>
		/// <param name="normalName">the name which may be prefixed with a namespace name</param>
		/// <returns>the bare name</returns>
        public static string ExtractName(string normalName)
        {
            string result = null;
            int ix = normalName.IndexOf(NamespaceRightBracket);
            if (0 < ix)
            {
                result = normalName.Substring(ix + NamespaceRightBracket.Length);
            }
            else
            {
				ix = normalName.IndexOf(NamespaceDelimiter);
				if (0 < ix)
				{
					result = normalName.Substring(ix + NamespaceDelimiter.Length);
				}
				else
				{
					result = normalName;
				}
            }
            return result;
        }

		/// <summary>
		/// Parse a source string into a namespace name and a name in that namespace.
		/// </summary>
		/// <param name="source">the string to be parsed</param>
		/// <param name="inModel">a model that may already or will contain the namespace name</param>
		/// <param name="space">the namespace name</param>
		/// <param name="name">the name in the namespace</param>
		public static void DecodeString(string source, OwlModel inModel, out string space, out string name)
		{
			if (null == inModel)
			{
				space = "";
				name = source;
			}
			else
			{
				int ix = source.IndexOf(NamespaceDelimiter);
				if (0 < ix) // Names that start with # are in current namespace.
				{
					// The source has both a namespace and a name in that namespace.
					space = NormalizeNamespace(source.Substring(0, ix + NamespaceDelimiter.Length));
					inModel.FindOrMakeNamespace(space);
					name = source.Substring(ix + NamespaceDelimiter.Length);
					if (name.StartsWith(NamespaceRightBracket))
					{
						name = name.Substring(NamespaceRightBracket.Length);
					}
				}
				else
				{
					space = inModel.DefaultNamespace;
					if (0 > ix)
					{
						name = source;
					}
					else
					{
						name = source.Substring(NamespaceDelimiter.Length);
					}
				}
			}
		}

		public static string EncodeString(string space, string name)
		{
			string result = null;
			if ((space.EndsWith(NamespaceDelimiter)) && (name.StartsWith(NamespaceDelimiter)))
			{
				result = space + name.Substring(NamespaceDelimiter.Length);
			}
			else
			{
				result = space + name;
			}
			return result;
		}

	}
}
