﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;

namespace wol
{
    public class ObjectProperty : Property
    {
		public const string OwlAsymmetricPropertyType = "&owl;AsymmetricProperty";
		public const string OwlInverseOf = "inverseOf";
		public const string OwlIrreflexivePropertyType = "&owl;IrreflexiveProperty";

        // rdf:resource attribute of rdfs:subPropertyOf element in owl:ObjectProperty element
        public ObjectProperty SubpropertyOf { get; set; }

        // rdf:resource attribute of owl:inverseOf element in owl:ObjectProperty element
        public ObjectProperty InverseOf { get; set; }

        // rdf:resource attribute of rdfs:range element in owl:ObjectProperty element
        // or
        // owl:Class element in rdfs:range element
        public OwlClass Range { get; set; }

        // "&owl;AsymmetricProperty" value of rdf:resource attribute of rdf:type element in owl:ObjectProperty element
        public bool? Asymmetric { get; set; }

        // "&owl;IrreflexiveProperty" value of rdf:resource attribute of rdf:type element in owl:ObjectProperty element
        public bool? Irreflexive { get; set; }

        protected override bool DeserializeInner(XElement source)
        {
            bool result = base.DeserializeInner(source);
            if (XmlHelp.ElementIs(source, OwlModel.RdfsSpace, RdfsRange))
            {
                Range = GetClassSpecification(source);
                result = true;
			} else {
				if (XmlHelp.ElementIs(source, OwlModel.RdfsSpace, RdfsSubPropertyOf))
				{
					SubpropertyOf = GetSubpropertySpecification(source);
					result = true;
				}
				else
				{
					if (XmlHelp.ElementIs(source, OwlModel.RdfSpace, RdfType))
					{
                        XAttribute resourceAttribute = XmlHelp.AttributeOf(source, OwlModel.RdfSpace, RdfResource);
                        if (null != resourceAttribute)
                        {
                            if (resourceAttribute.Value == OwlAsymmetricPropertyType)
                            {
                                Asymmetric = true;
                            }
                            if (resourceAttribute.Value == OwlIrreflexivePropertyType)
                            {
                                Irreflexive = true;
                            }
                        }
						result = true;
					}
					else
					{
						if (XmlHelp.ElementIs(source, OwlModel.OwlSpace, OwlInverseOf))
						{
							InverseOf = GetSubpropertySpecification(source);
							result = true;
						}
					}
				}
            }
            return result;
        }

        public override string Display
        {
            get
            {
                string result = "Object" + base.Display;
                if (null != Range)
                {
					result += System.Environment.NewLine + "\trange:" + Range.DisplayNameOrAll;
                }
                if (null != Asymmetric)
                {
                    result += System.Environment.NewLine + "\tasymmetric=" + Asymmetric.ToString();
                }
                if (null != Irreflexive)
                {
					result += System.Environment.NewLine + "\tirreflexive=" + Irreflexive.ToString();
                }
                if (null != SubpropertyOf)
                {
					result += System.Environment.NewLine + "\tsubpropertyOf:" + SubpropertyOf.DisplayNameOrAll;
                }
                if (null != InverseOf)
                {
					result += System.Environment.NewLine + "\tinverseOf:" + InverseOf.DisplayNameOrAll;
                }
                return result;
            }
        }

		protected ObjectProperty GetSubpropertySpecification(XElement source)
		{
			ObjectProperty result = null;
			XAttribute resourceAttribute = XmlHelp.AttributeOf(source, OwlModel.RdfSpace, RdfResource);
			if (null != resourceAttribute)
			{
				string normalName = OwlNameInSpace.NormalizeName(resourceAttribute.Value, InModel);
				result = FindOrMakeObjectProperty(normalName);
			}
			return result;
		}

		protected ObjectProperty FindOrMakeObjectProperty(string name)
		{
			ObjectProperty result = null;
			if (null != InModel)
			{
				result = InModel.FindOrMakeObjectProperty(name);
			}
			return result;
		}

		protected override void ChangedName(string oldName)
		{
			InModel.ObjectPropertyChangedName(oldName, this);
		}

    }
}
